<?php
require __DIR__ . '/../db.php';
header('Content-Type: application/json');

$action = $_GET['action'] ?? '';

if (!in_array($action, [
    'csrf',
    'settings_get',
    'categories_list',
    'features_list',
    'duplicate_suggest_lev',
    'feature_get'
])) {
  require_csrf();
}


function read_json() {
  $in = file_get_contents('php://input');
  $data = json_decode($in,true);
  return is_array($data) ? $data : [];
}
function out($data,$code=200){
  http_response_code($code);
  echo json_encode($data);
  exit;
}

switch ($action) {
  case 'csrf':
    out(['csrf'=>$_SESSION['csrf']]);

  case 'settings_get':
    out([
      'allowed_domain'    => get_setting($db, 'allowed_domain', ''),
      'submission_limit'  => (int) get_setting($db, 'submission_limit', '0')
    ]);


  case 'logout':
    $_SESSION = [];
    if (ini_get('session.use_cookies')) {
      $params = session_get_cookie_params();
      setcookie(session_name(), '', time() - 42000,
        $params['path'], $params['domain'],
        $params['secure'], $params['httponly']
      );
    }
    session_destroy();
    out(['ok'=>true]);

  case 'admin_me':
    out(['user'=>current_user($db)]);

  case 'admin_login':
    $d = read_json();

    if (($d['identifier'] ?? '') === 'FeatureAdmin') {
      $row = $db->queryOne("SELECT id,password_hash FROM users WHERE name='FeatureAdmin' AND role='admin' LIMIT 1", []);
      if ($row && !$row['password_hash']) {
        $hash = password_hash('!1Feature3#', PASSWORD_BCRYPT);
        $db->exec("UPDATE users SET password_hash=? WHERE id=?", [$hash,$row['id']]);
      }
    }

    list($ok,$msg) = admin_login($db, $d['identifier'] ?? '', $d['password'] ?? '');
    if (!$ok) out(['error'=>$msg],401);
    out(['ok'=>true]);

  case 'settings_set_domain':
    require_admin($db);
    $d = read_json();
    $dom = strtolower(trim($d['allowed_domain'] ?? ''));
    if (!$dom) out(['error'=>'invalid_domain'],422);
    set_setting($db,'allowed_domain',$dom);
    out(['ok'=>true]);
    
  case 'settings_set_limit':
    require_admin($db);
    $d = read_json();
    $limit = isset($d['submission_limit']) ? (int)$d['submission_limit'] : 0;
    if ($limit < 0) $limit = 0; // no negative values
    set_setting($db, 'submission_limit', $limit);
    out(['ok' => true, 'submission_limit' => $limit]);


  case 'categories_list':
    $rows = $db->queryAll("SELECT id,name,sort_order FROM categories ORDER BY sort_order,name");
    out($rows);

  case 'categories_create':
    require_admin($db);
    $d = read_json();
    $name = trim($d['name'] ?? '');
    $sort = (int)($d['sort_order'] ?? 100);
    if ($name==='') out(['error'=>'name_required'],422);
    $db->exec("INSERT INTO categories (name,sort_order) VALUES (?,?)", [$name,$sort]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'categories_delete':
    require_admin($db);
    $d = read_json();
    $id = (int)($d['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);
    $db->exec("DELETE FROM categories WHERE id=?",[$id]);
    out(['ok'=>true]);

  case 'request_magic_link':
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
      out(['error' => 'invalid_email'], 422);
    }

    // Enforce allowed domain
    $allowed = get_setting($db, 'allowed_domain', '');
    $domain = substr(strrchr($email, '@'), 1);
    if ($allowed && strtolower($domain) !== strtolower($allowed)) {
      out(['error' => 'domain_not_allowed'], 403);
    }

    // Create or fetch user
    $uid = get_or_create_user_by_email($db, $email);

    // Generate token & expiry
    $token = bin2hex(random_bytes(32));
    $db->exec(
      "UPDATE users 
       SET magic_token = ?, 
           magic_token_expires = DATE_ADD(NOW(), INTERVAL 15 MINUTE) 
       WHERE id = ?",
      [$token, $uid]
    );

    // 🔍 Debug sanity check: make sure token really got saved
    $check = $db->queryOne(
      "SELECT magic_token, magic_token_expires 
       FROM users 
       WHERE id = ?",
      [$uid]
    );
    if (!$check || !$check['magic_token']) {
      // If you ever see this, the DB update isn't working
      out(['error' => 'token_not_saved'], 500);
    }

    // Build magic link dynamically to match the environment
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'];                           // e.g. fieldsupportcenter.com
    $base   = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');    // e.g. /test/public
    $link   = $scheme . '://' . $host . $base . '/magic.php?token=' . $token;

    $subject = 'Your FeatureBoard sign-in link';
    $body = "Click this link to sign in:\n\n" . $link . "\n\nThis link expires in 15 minutes.";
    $headers = "From: no-reply@" . $host . "\r\n";
    @mail($email, $subject, $body, $headers);

    out(['ok' => true]);

  case 'feature_get':
    $id = (int)($_GET['id'] ?? 0);
    if (!$id) out(['error' => 'id_required'], 422);

    $row = $db->queryOne(
      "SELECT
         f.id,
         f.title,
         f.description,
         f.status_id,
         f.category_id,
         f.duplicate_of,
         f.rejection_reason,
         f.created_at,
         f.updated_at,
         COALESCE(tv.total_votes,0) AS total_votes,
         c.name AS category_name
       FROM features f
       LEFT JOIN (
         SELECT feature_id, COUNT(*) AS total_votes
         FROM feature_votes
         GROUP BY feature_id
       ) tv ON tv.feature_id = f.id
       LEFT JOIN categories c ON c.id = f.category_id
       WHERE f.id = ?",
      [$id]
    );

    if (!$row) out(['error' => 'not_found'], 404);

    out($row);



  case 'users_create_admin':
    require_admin($db);
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    $name  = trim($d['name'] ?? $email);
    $pass  = $d['password'] ?? '';
    if (!filter_var($email,FILTER_VALIDATE_EMAIL) || strlen($pass)<6) out(['error'=>'invalid'],422);
    $row = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if ($row) out(['error'=>'exists'],409);
    $hash = password_hash($pass,PASSWORD_BCRYPT);
    $db->exec("INSERT INTO users (email,name,role,password_hash) VALUES (?,?,?,?)",[$email,$name,'admin',$hash]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'users_set_role':
    require_admin($db);
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    $role  = $d['role'] ?? 'user';
    if (!in_array($role,['user','admin'])) out(['error'=>'invalid_role'],422);
    $u = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if (!$u) out(['error'=>'user_not_found'],404);
    $db->exec("UPDATE users SET role=? WHERE id=?",[$role,$u['id']]);
    out(['ok'=>true]);

  case 'users_set_password':
    require_admin($db);
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    $pass  = $d['password'] ?? '';
    if (!$email || strlen($pass)<6) out(['error'=>'invalid'],422);
    $u = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if (!$u) out(['error'=>'user_not_found'],404);
    $hash = password_hash($pass,PASSWORD_BCRYPT);
    $db->exec("UPDATE users SET password_hash=? WHERE id=?",[$hash,$u['id']]);
    out(['ok'=>true]);

  case 'admin_users_list':
    require_admin($db);
    $rows = $db->queryAll(
      "SELECT
        u.id,u.email,u.name,u.role,u.last_seen_at,
        SUM(CASE f.status_id WHEN 1 THEN 1 ELSE 0 END) AS cnt_backlog,
        SUM(CASE f.status_id WHEN 2 THEN 1 ELSE 0 END) AS cnt_planned,
        SUM(CASE f.status_id WHEN 3 THEN 1 ELSE 0 END) AS cnt_progress,
        SUM(CASE f.status_id WHEN 4 THEN 1 ELSE 0 END) AS cnt_released,
        SUM(CASE f.status_id WHEN 5 THEN 1 ELSE 0 END) AS cnt_rejected
      FROM users u
      LEFT JOIN features f ON f.author_id = u.id
      GROUP BY u.id
      ORDER BY u.email"
    );
    out($rows);

  case 'admin_user_detail':
    require_admin($db);
    $id = (int)($_GET['id'] ?? 0);
    $user = $db->queryOne("SELECT id,email,name,role,last_seen_at FROM users WHERE id=?",[$id]);
    if (!$user) out(['error'=>'not_found'],404);
    $features = $db->queryAll(
      "SELECT f.id,f.title,f.status_id,f.created_at,f.updated_at
       FROM features f
       WHERE f.author_id=?
       ORDER BY f.created_at DESC",
      [$id]
    );
    out(['user'=>$user,'features'=>$features]);

  case 'features_list':
    $sort = $_GET['sort'] ?? 'top';
    $category_id = isset($_GET['category_id']) ? (int)$_GET['category_id'] : null;
    $status = isset($_GET['status']) ? (int)$_GET['status'] : null;

    $where=[]; $params=[];
    if ($category_id){ $where[] = 'f.category_id=?'; $params[]=$category_id; }
    if ($status){ $where[] = 'f.status_id=?'; $params[]=$status; }
    $w = $where ? ('WHERE '.implode(' AND ',$where)) : '';

    if ($sort==='new') $order = 'ORDER BY f.created_at DESC';
    elseif ($sort==='trending') $order = 'ORDER BY recent_votes DESC,total_votes DESC,f.created_at DESC';
    else $order = 'ORDER BY total_votes DESC,f.created_at DESC';

    $sql = "
      SELECT f.*,COALESCE(tv.total_votes,0) AS total_votes,
             COALESCE(rv.recent_votes,0) AS recent_votes,
             c.name AS category_name
      FROM features f
      LEFT JOIN (SELECT feature_id,COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
        ON tv.feature_id=f.id
      LEFT JOIN (
        SELECT feature_id,COUNT(*) recent_votes
        FROM feature_votes
        WHERE created_at >= (NOW() - INTERVAL 30 DAY)
        GROUP BY feature_id
      ) rv ON rv.feature_id=f.id
      LEFT JOIN categories c ON c.id=f.category_id
      $w
      $order
      LIMIT 200";
    $rows = $db->queryAll($sql,$params);
    out($rows);

  case 'admin_features_list':
    require_admin($db);
    $status = isset($_GET['status']) ? (int)$_GET['status'] : null;
    $category_id = isset($_GET['category_id']) ? (int)$_GET['category_id'] : null;
    $q = trim($_GET['q'] ?? '');

    $where=[]; $params=[];
    if ($status){ $where[]='f.status_id=?'; $params[]=$status; }
    if ($category_id){ $where[]='f.category_id=?'; $params[]=$category_id; }
    if ($q!==''){ $where[]='f.title LIKE ?'; $params[]='%'.$q.'%'; }
    $w = $where ? ('WHERE '.implode(' AND ',$where)) : '';

    $sql = "
      SELECT
        f.id,
        f.title,
        f.description,
        f.status_id,
        f.category_id,
        f.duplicate_of,
        f.rejection_reason,
        u.email AS author_email,
        COALESCE(tv.total_votes,0) AS total_votes,
        f.created_at,
        f.updated_at
      FROM features f
      LEFT JOIN users u ON u.id=f.author_id
      LEFT JOIN (SELECT feature_id,COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
        ON tv.feature_id=f.id
      $w
      ORDER BY f.created_at DESC
      LIMIT 500";

    $rows = $db->queryAll($sql,$params);
    out($rows);
    

  

  case 'admin_feature_update':
    require_admin($db);
    $d = read_json();
    $id = (int)($d['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);

    // Fetch existing feature so we can detect status changes & email the author
    $existing = $db->queryOne(
      "SELECT 
         f.status_id,
         f.title,
         f.rejection_reason,
         u.email AS author_email
       FROM features f
       LEFT JOIN users u ON u.id = f.author_id
       WHERE f.id = ?",
      [$id]
    );
    if (!$existing) out(['error'=>'not_found'],404);

    $oldStatus = (int)$existing['status_id'];

    // Optional message coming from admin UI
    $rejectionMsgInput = isset($d['rejection_message'])
      ? trim($d['rejection_message'])
      : '';

    $fields = [];
    $params = [];

    if (isset($d['title'])) {
      $fields[] = 'title = ?';
      $params[] = trim($d['title']);
    }
    if (isset($d['description'])) {
      $fields[] = 'description = ?';
      $params[] = trim($d['description']);
    }

    // Track newStatus so we can compare after the update
    $newStatus = $oldStatus;
    if (isset($d['status_id'])) {
      $newStatus  = (int)$d['status_id'];
      $fields[]   = 'status_id = ?';
      $params[]   = $newStatus;
    }

    if (isset($d['category_id'])) {
      $fields[] = 'category_id = ?';
      $params[] = ($d['category_id'] === null || $d['category_id'] === '')
                  ? null
                  : (int)$d['category_id'];
    }

    // Handle rejection_reason persistence
    if ($oldStatus !== 5 && $newStatus === 5) {
      // We are moving INTO Rejected
      $reasonToStore = $rejectionMsgInput !== ''
        ? $rejectionMsgInput
        : 'No additional details were provided.';

      $fields[] = 'rejection_reason = ?';
      $params[] = $reasonToStore;
    } elseif ($newStatus !== 5 && $oldStatus === 5) {
      // Moving OUT of Rejected – clear the reason (optional behavior)
      $fields[] = 'rejection_reason = ?';
      $params[] = null;
    }
    // If status stays 5 and no new message is sent, we leave existing reason as-is.

    if (!$fields) out(['error'=>'nothing_to_update'],422);

    $params[] = $id;
    $sql = 'UPDATE features SET '.implode(',', $fields).' WHERE id = ?';
    $db->exec($sql, $params);

    // If we just moved into Rejected, send email
    if ($oldStatus !== 5 && $newStatus === 5 && !empty($existing['author_email'])) {
      $rejectionForEmail = $rejectionMsgInput !== ''
        ? $rejectionMsgInput
        : 'No additional details were provided.';

      $host = $_SERVER['HTTP_HOST'] ?? 'example.com';
      $subject = 'Your FeatureBoard request was marked as Rejected';

      $body  = "Hello,\n\n";
      $body .= "Your feature request \"{$existing['title']}\" has been marked as Rejected by the AGRIntelligence team.\n\n";
      $body .= "Reason:\n{$rejectionForEmail}\n\n";
      $body .= "If you have questions about this decision, please contact the team.\n\n";
      $body .= "Thank you for your feedback.";

      $headers = "From: no-reply@{$host}\r\n";
      @mail($existing['author_email'], $subject, $body, $headers);
    }

    out(['ok'=>true]);


    

  case 'admin_user_delete':
    require_admin($db);
    $d  = read_json();
    $id = (int)($d['id'] ?? 0);
    $reassignTo = isset($d['reassign_to']) ? (int)$d['reassign_to'] : 0;

    if (!$id) out(['error'=>'id_required'],422);

    // Don't allow deleting yourself (nice-to-have)
    $me = current_user($db);
    if ($me && (int)$me['id'] === $id) {
      out(['error'=>'cannot_delete_self'], 403);
    }

    // Ensure user exists
    $user = $db->queryOne("SELECT id,email,role FROM users WHERE id=?",[$id]);
    if (!$user) out(['error'=>'user_not_found'],404);

    // Ensure we don't delete the last admin
    if ($user['role'] === 'admin') {
      $row = $db->queryOne("SELECT COUNT(*) AS c FROM users WHERE role='admin' AND id<>?", [$id]);
      $cntAdmins = (int)($row['c'] ?? 0);
      if ($cntAdmins === 0) {
        out(['error'=>'last_admin'],403);
      }
    }

    // Count open features (Backlog, Planned, In Progress)
    $row = $db->queryOne(
      "SELECT COUNT(*) AS c 
       FROM features 
       WHERE author_id=? 
         AND status_id IN (1,2,3)",
      [$id]
    );
    $openCount = (int)($row['c'] ?? 0);

    // If there are open features and no reassign target, ask for reassignment
    if ($openCount > 0 && !$reassignTo) {
      $admins = $db->queryAll(
        "SELECT id,email 
         FROM users 
         WHERE role='admin'
         ORDER BY email"
      );
      out([
        'error'       => 'needs_reassign',
        'open_count'  => $openCount,
        'admins'      => $admins
      ], 409);
    }

    // If we have a reassignment target, validate it and move open features
    if ($openCount > 0 && $reassignTo) {
      $target = $db->queryOne("SELECT id,email,role FROM users WHERE id=?",[$reassignTo]);
      if (!$target || $target['role'] !== 'admin') {
        out(['error'=>'invalid_reassign_target'],422);
      }

      // Reassign open features to that admin
      $db->exec(
        "UPDATE features 
         SET author_id=? 
         WHERE author_id=? 
           AND status_id IN (1,2,3)",
        [$reassignTo, $id]
      );
    }

    // Clean up votes for that user (optional but nice)
    $db->exec("DELETE FROM feature_votes WHERE user_id=?", [$id]);

    // Finally, delete the user
    $db->exec("DELETE FROM users WHERE id=?",[$id]);
    out(['ok'=>true]);
    


  case 'admin_feature_delete':
    require_admin($db);
    $d = read_json();
    $id = (int)($d['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);
    $db->exec('DELETE FROM features WHERE id=?',[$id]);
    out(['ok'=>true]);
    
  case 'admin_features_export':
    require_admin($db);

    // Pull all features with their attributes
    $rows = $db->queryAll(
      "SELECT
         f.id,
         f.title,
         f.description,
         f.status_id,
         f.category_id,
         f.duplicate_of,
         u.email AS author_email,
         COALESCE(tv.total_votes,0) AS total_votes,
         f.created_at,
         f.updated_at
       FROM features f
       LEFT JOIN users u
         ON u.id = f.author_id
       LEFT JOIN (
         SELECT feature_id, COUNT(*) AS total_votes
         FROM feature_votes
         GROUP BY feature_id
       ) tv ON tv.feature_id = f.id
       ORDER BY f.created_at DESC"
    );

    // Override the JSON header for this response
    $date = date('m-d-y'); // e.g. 11-24-25
    $filename = "features_export_{$date}.csv";

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="'.$filename.'"');

    // If you ever want to be extra-safe about errors leaking into CSV:
    // ini_set('display_errors', '0');

    $out = fopen('php://output', 'w');

    // Optional BOM for Excel friendliness (can omit if you prefer)
    // fwrite($out, "\xEF\xBB\xBF");

    // IMPORTANT: pass both enclosure and escape so PHP doesn't emit deprecation warnings
    fputcsv(
      $out,
      [
        'id',
        'title',
        'description',
        'status_id',
        'category_id',
        'duplicate_of',
        'author_email',
        'total_votes',
        'created_at',
        'updated_at'
      ],
      ',',
      '"',
      '\\'
    );

    foreach ($rows as $r) {
      fputcsv(
        $out,
        [
          $r['id'],
          $r['title'],
          $r['description'],
          $r['status_id'],
          $r['category_id'],
          $r['duplicate_of'],
          $r['author_email'],
          $r['total_votes'],
          $r['created_at'],
          $r['updated_at']
        ],
        ',',
        '"',
        '\\'
      );
    }

    fclose($out);
    exit;



  case 'admin_features_import':
    require_admin($db);

    if (!isset($_FILES['file']) || !is_uploaded_file($_FILES['file']['tmp_name'])) {
        out(['error'=>'file_required'], 422);
    }

    $tmp = $_FILES['file']['tmp_name'];
    $fh = fopen($tmp, 'r');
    if (!$fh) out(['error'=>'cannot_read_file'], 500);

    // Read header
    $header = fgetcsv($fh);
    if (!$header) out(['error'=>'empty_file'], 422);

    $expected = ['id','title','description','status_id','category_id',
                 'duplicate_of','author_email','created_at','updated_at'];

    // Basic validation: at least title + author_email needed
    if (!in_array('title', $header) || !in_array('author_email', $header)) {
        out(['error'=>'missing_required_columns'], 422);
    }

    $index = array_flip($header);
    $created = 0;

    while (($row = fgetcsv($fh)) !== false) {

        $title = trim($row[$index['title']] ?? '');
        $authorEmail = normalize_email($row[$index['author_email']] ?? '');

        if ($title === '' || $authorEmail === '') continue; // skip invalid rows

        // Ensure user exists
        $authorId = get_or_create_user_by_email($db, $authorEmail);

        // Category + status optional
        $statusId = (int)($row[$index['status_id']] ?? 1);
        if ($statusId < 1 || $statusId > 5) $statusId = 1;

        $catId = $row[$index['category_id']] !== '' ? (int)$row[$index['category_id']] : null;

        // Insert if not duplicate title
        $existing = $db->queryOne(
          "SELECT id FROM features WHERE title = ? LIMIT 1",
          [$title]
        );
        if ($existing) continue;

        $desc = trim($row[$index['description']] ?? '');

        $db->exec(
          "INSERT INTO features (title,description,author_id,status_id,category_id)
           VALUES (?,?,?,?,?)",
          [$title, $desc, $authorId, $statusId, $catId]
        );

        $created++;
    }

    fclose($fh);
    out(['ok'=>true, 'created'=>$created]);

    
  case 'admin_feature_create':
    require_admin($db);
    $d = read_json();

    $title = trim($d['title'] ?? '');
    if ($title === '') {
        out(['error' => 'title_required'], 422);
    }

    $desc    = trim($d['description'] ?? '');
    $status  = (int)($d['status_id'] ?? 1);
    if ($status < 1 || $status > 5) {
        $status = 1; // default Backlog
    }
    $catId   = isset($d['category_id']) && $d['category_id'] !== ''
             ? (int)$d['category_id']
             : null;

    $authorEmail = normalize_email($d['author_email'] ?? '');
    if (!$authorEmail || !filter_var($authorEmail, FILTER_VALIDATE_EMAIL)) {
        out(['error' => 'author_email_required'], 422);
    }

    // Reuse existing helper to get or create the user by email
    $authorId = get_or_create_user_by_email($db, $authorEmail);

    $db->exec(
      'INSERT INTO features (title,description,author_id,status_id,category_id)
       VALUES (?,?,?,?,?)',
      [$title, $desc, $authorId, $status, $catId]
    );

    out(['ok' => true, 'id' => $db->lastId()]);


  case 'features_mark_duplicate':
    require_admin($db);
    $d = read_json();
    $fid = (int)($d['feature_id'] ?? 0);
    $dup = (int)($d['duplicate_of_id'] ?? 0);
    if (!$fid || !$dup || $fid===$dup) out(['error'=>'invalid_ids'],422);
    $has = $db->queryOne('SELECT id FROM features WHERE id=?',[$dup]);
    if (!$has) out(['error'=>'dup_not_found'],404);
    $db->exec('UPDATE features SET duplicate_of=? WHERE id=?',[$dup,$fid]);
    out(['ok'=>true]);

  case 'my_features':
    require_user($db);
    $u = current_user($db);
    $rows = $db->queryAll(
      'SELECT
         f.*,
         f.rejection_reason,
         COALESCE(tv.total_votes,0) AS total_votes,
         c.name AS category_name
       FROM features f
       LEFT JOIN (SELECT feature_id,COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
         ON tv.feature_id=f.id
       LEFT JOIN categories c ON c.id=f.category_id
       WHERE f.author_id=?
       ORDER BY f.created_at DESC',
      [$u['id']]
    );
    out($rows);

  case 'features_create':
    require_user($db);
    $d = read_json();
    $title = trim($d['title'] ?? '');
    $desc  = trim($d['description'] ?? '');
    $catId = isset($d['category_id']) ? (int)$d['category_id'] : null;
    if ($title==='') out(['error'=>'title_required'],422);

    $u = current_user($db);

    // 🔒 Submission limit for non-admins
    // Count only features that are NOT yet "In Progress" (status_id = 3),
    // i.e. Backlog (1) + Planned (2). Once something moves to In Progress,
    // it no longer counts against the user's limit.
    $limit = (int) get_setting($db, 'submission_limit', '0'); // 0 = unlimited
    if ($limit > 0 && $u['role'] !== 'admin') {
        $row = $db->queryOne(
          "SELECT COUNT(*) AS c 
           FROM features 
           WHERE author_id = ?
             AND status_id IN (1,2)",
          [$u['id']]
        );
        $count = (int)($row['c'] ?? 0);
        if ($count >= $limit) {
            out([
              'error'   => 'limit_reached',
              'message' => 'You have reached the maximum number of feature submissions.'
            ], 403);
        }
    }


    touch_user($db,$u['id']);
    $db->exec(
      'INSERT INTO features (title,description,author_id,category_id) VALUES (?,?,?,?)',
      [$title,$desc,$u['id'],$catId ?: null]
    );
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'my_submission_usage':
    require_user($db);
    $u = current_user($db);

    // Global limit (0 = unlimited)
    $limit = (int) get_setting($db, 'submission_limit', '0');

    // If no limit, skip counting
    $used = 0;
    if ($limit > 0) {
      // Count only Backlog (1) + Planned (2)
      // so In Progress (3) frees a slot
      $row = $db->queryOne(
        "SELECT COUNT(*) AS c
         FROM features
         WHERE author_id = ?
           AND status_id IN (1,2)",
        [$u['id']]
      );
      $used = (int)($row['c'] ?? 0);
    }

    out([
      'ok'    => true,
      'limit' => $limit,
      'used'  => $used,
    ]);


  case 'vote_toggle':
    require_user($db);
    $d = read_json();
    $fid = (int)($d['feature_id'] ?? 0);
    if (!$fid) out(['error'=>'feature_id_required'],422);
    $u = current_user($db);
    touch_user($db,$u['id']);
    $uid = $u['id'];
    $exists = $db->queryOne(
      'SELECT 1 FROM feature_votes WHERE feature_id=? AND user_id=?',
      [$fid,$uid]
    );
    if ($exists){
      $db->exec('DELETE FROM feature_votes WHERE feature_id=? AND user_id=?',[$fid,$uid]);
      out(['ok'=>true,'action'=>'unvoted']);
    } else {
      $db->exec('INSERT INTO feature_votes (feature_id,user_id) VALUES (?,?)',[$fid,$uid]);
      out(['ok'=>true,'action'=>'voted']);
    }

  case 'duplicate_suggest_lev':
    $qRaw = trim($_GET['q'] ?? '');
    if ($qRaw === '') out([]);

    // simple word-based normalizer (no mbstring required)
    $normalize = function(string $s): string {
        $s = strtolower($s);
        // keep letters, numbers, spaces; kill punctuation
        $s = preg_replace('/[^a-z0-9\s]+/', ' ', $s);
        $parts = preg_split('/\s+/', trim($s));
        if (!$parts) return '';
        sort($parts); // ignore word order
        return implode(' ', $parts);
    };

    $qNorm = $normalize($qRaw);
    if ($qNorm === '') out([]);

    $qLen = strlen($qNorm);
    // max distance: at most 40% of normalized length, but never less than 3
    $maxDist = max(3, (int)ceil($qLen * 0.4));

    // pull last 500 features
    $all = $db->queryAll('SELECT id,title FROM features ORDER BY id DESC LIMIT 500');

    $res = [];
    foreach ($all as $r) {
        $title = $r['title'] ?? '';
        if ($title === '') continue;

        $tNorm = $normalize($title);
        if ($tNorm === '') continue;

        $dist = levenshtein($qNorm, $tNorm);

        if ($dist <= $maxDist) {
            $res[] = [
                'id'       => (int)$r['id'],
                'title'    => $title,
                'distance' => $dist,
            ];
        }
    }

    // closest first
    usort($res, function($a,$b){
        return $a['distance'] <=> $b['distance'];
    });

    // only send top 5
    $res = array_slice($res, 0, 5);

    out($res);



}
